<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 27.01.14 11:21
 */

namespace VM\AdminBundle\Services;


use Doctrine\ORM\EntityManager;
use VM\ApiBundle\Entity\Company;
use VM\ApiBundle\Entity\CompanyReport;
use VM\ApiBundle\Entity\GroupRoundReportCode;
use VM\ApiBundle\Entity\Round;
use VM\ApiBundle\Entity\User;
use VM\FinancialStatementsBundle\Services\AccountChanges\AccountChanges;
use VM\FinancialStatementsBundle\Services\FinancialStatementGame;
use VM\FinancialStatementsBundle\Services\Reports\FullReport;
use VM\FinancialStatementsBundle\Services\Reports\SimpleReport;
use VM\PackageGeneratorBundle\Services\Export\ExportProcessCreator;

class EndRoundService
{

    /**
     * @var ApiGetter
     */
    protected $ApiGetter;

    /**
     * @var AccountChanges
     */
    protected $AccountChanges;

    /**
     * @var FinancialStatementGame
     */
    protected $FinancialStatementGame;


    /**
     * @var ExportProcessCreator
     */
    protected $ExportProcessCreator;

    /**
     * @var AuthCommunication
     */
    protected $AuthCommunication;

    /**
     * @var RDFExportGeneratorService
     */
    protected $RDFExportGeneratorService;


    protected $jappsExport = false;


    public function __construct(ApiGetter $ApiGetter, AccountChanges $AccountChanges, FinancialStatementGame $FinancialStatementGame, ExportProcessCreator $ExportProcessCreator, AuthCommunication $AuthCommunication, RDFExportGeneratorService $RDFExportGeneratorService, $jappsExport = false)
    {
        $this->ApiGetter = $ApiGetter;
        $this->AccountChanges = $AccountChanges;
        $this->FinancialStatementGame = $FinancialStatementGame;
        $this->ExportProcessCreator = $ExportProcessCreator;
        $this->AuthCommunication = $AuthCommunication;
        $this->RDFExportGeneratorService = $RDFExportGeneratorService;
        $this->jappsExport = $jappsExport;
    }



    public function endRound(Round $CurrentRound, $createAccountChanges = true)
    {
        $Group = $this->ApiGetter->getGroup($CurrentRound->getUserGroupId());

        $dir = __DIR__ . '/../../../../web/reports/';

        $subDir = $dir . 'group_' . $Group->getId() . '/round_' . $CurrentRound->getId() . '/';
        $Company = $this->ApiGetter->getGroupCompanyOrFalse($Group);

        if($Company instanceof Company)
        {
            if(!is_dir($subDir))
            {
                mkdir($subDir, 0777, true);
            }

            $data = $this->FinancialStatementGame->statement($CurrentRound,$Company);
            $dataFull = json_decode(json_encode($this->FinancialStatementGame->getFullData()));
            $dataFull->companyName = $Company->getName();
            $dataFull->grupName = $Group->getName();
            $data->companyName = $Company->getName();
            $data->grupName = $Group->getName();

            $rdfExports = $this->RDFExportGeneratorService;
            $rdfExports->generateRDFExportOfSimpleFinancialReport($data, $Company, $CurrentRound);

            if($createAccountChanges)
            {
                $this->AccountChanges->setProductionChanges($this->FinancialStatementGame->getProductionData());

                $this->AccountChanges->setCompany($Company);
                $this->AccountChanges->generateChanges();
            }


            $SimpleReport = new SimpleReport($data);
            $path = $subDir . 'simple_report.pdf';
            $SimpleReport->getPdf()->Output($path, 'F');

            $title = 'Raport uproszczony (' . $CurrentRound->getName() . ') - ' . $Group->getName();
            $this->publishReport($CurrentRound, $Company, $title, $path, $subDir, $Group);

            $FullReport = new FullReport($dataFull, $CurrentRound->getSequence());
            $path = $subDir . 'full_report.pdf';
            $FullReport->getPdf()->Output($path, 'F');

            $title = 'Pełny raport (' . $CurrentRound->getName() . ') - ' . $Group->getName();
            $this->publishReport($CurrentRound, $Company, $title, $path, $subDir, $Group, true);

        }

        $NextRound = $this->ApiGetter->getNextRoundOrFalse($CurrentRound);
        $NextRound->setStartAt(new \DateTime());
    }


    /**
     * @return array
     */
    public function getRoundsToEnd()
    {
        $rounds = array();
        $Now = new \DateTime();

        $QueryBuilder = $this->ApiGetter->getEntityManager()->getRepository('VMApiBundle:Round')->createQueryBuilder('r');

        $QueryBuilder->andWhere('r.isDone = :isDone')
            ->andWhere('r.isCounting = :isCounting')
            ->andWhere('r.startAt <= :now')
            ->setParameter('isCounting', false)
            ->setParameter('isDone', false)
            ->setParameter('now', $Now)
            ->orderBy('r.sequence', 'ASC')

        ;

        $foundRounds = $QueryBuilder->getQuery()->getResult();


        /** @var Round $Round */
        foreach($foundRounds as $Round)
        {
            $NextRound = $this->ApiGetter->getNextRoundOrFalse($Round);
            $StartAt = (empty($NextRound)) ? null : $NextRound->getStartAt();
            if(!empty($StartAt) && $StartAt->format("Y-m-d H:i:s") < $Now->format("Y-m-d H:i:s"))
            {
                $rounds[] = $Round;
            }
        }


        return $rounds;
    }


    /**
     * @param Round $Round
     * @return bool|Round
     */
    public function getPreviousRound(Round $Round)
    {
        return $this->ApiGetter->getPreviousRoundOrFalse($Round);
    }


    /**
     * @param Company $Company
     * @param Round $Round
     * @param $title
     * @param $response
     * @param bool $extendedReport
     * @return CompanyReport
     */
    protected function createCompanyReport(Company $Company, Round $Round, $title, $response, $extendedReport = false, GroupRoundReportCode $ProductCode = null)
    {
        if(empty($ProductCode))
        {
            $ProductCode = new GroupRoundReportCode();
            $ProductCode->setGroup($Company->getUserGroups());
            $ProductCode->setRound($Round);
            $ProductCode->setCode($response['productCode']);
            $ProductCode->setExtended($extendedReport);
            $ProductCode->setVersion(1);
            $this->ApiGetter->getEntityManager()->persist($ProductCode);
            $this->ApiGetter->getEntityManager()->flush();
        }

        $CompanyReport = new CompanyReport();
        $CompanyReport->setCompany($Company);
        $CompanyReport->setRound($Round);
        $CompanyReport->setTitle($title);
//        $CompanyReport->setVersion(1);
        $CompanyReport->setExtended($extendedReport);
        $CompanyReport->setProductCode($ProductCode);
        $this->ApiGetter->getEntityManager()->persist($CompanyReport);
        $this->ApiGetter->getEntityManager()->flush();

        return $CompanyReport;
    }


    /**
     * @param Round $CurrentRound
     * @param $Company
     * @param $title
     * @param $path
     * @param $subDir
     * @param $Group
     * @param bool $extendedReport
     * @throws \Exception
     */
    protected function publishReport(Round $CurrentRound, $Company, $title, $path, $subDir, $Group, $extendedReport = false)
    {
        $CompanyReport = $this->ApiGetter->getCompanyReportOrFalse($CurrentRound, $Company, $extendedReport);
        $GroupRoundReportCode = $this->ApiGetter->getAuthProductCodeForCompanyAndRoundOrFalse($Company, $CurrentRound, $extendedReport);

        if($CompanyReport === false)
        {
            if(empty($GroupRoundReportCode))
            {
                $productCategory = ($extendedReport) ? 'GAME_FULL_REPORT' : 'GAME_SIMPLE_REPORT';
                $response = $this->AuthCommunication->createProduct($title,$productCategory);
                if(!empty($response['error']))
                {
                    throw new \Exception($response['error']['message'], $response['error']['code']);
                }
                $CompanyReport = $this->createCompanyReport($Company, $CurrentRound, $title, $response, $extendedReport);
            }
            else
            {
                $response = array('productCode' => $GroupRoundReportCode->getCode());
                $CompanyReport = $this->createCompanyReport($Company, $CurrentRound, $title, $response, $extendedReport, $GroupRoundReportCode);
            }
        }
        else
        {
            $GroupRoundReportCode->setVersion($GroupRoundReportCode->getVersion() + 1);
            $CompanyReport->setTitle($title);
        }

        $GroupRoundReportCode = $this->ApiGetter->getAuthProductCodeForCompanyAndRoundOrFalse($Company, $CurrentRound, $extendedReport);

        $xmlZip = $this->ExportProcessCreator->process($path, $CompanyReport->getProductCode()->getCode(), $subDir, 'xml', $title, '', true);
        $jsonZip = $this->ExportProcessCreator->process($path, $CompanyReport->getProductCode()->getCode(), $subDir, 'json', $title, '', true);
        $mobileZip = $this->ExportProcessCreator->process($path, $CompanyReport->getProductCode()->getCode(), $subDir, 'mobile', $title, '', true);

        $dataDir = ($extendedReport) ? '1/data/' : '0/data/';
        if(!is_dir($subDir . $dataDir))
        {
            mkdir($subDir . $dataDir, 0777, true);
        }
        else
        {
            system('rm -rf ' . $subDir . $dataDir. '*');
        }

        $ZipArchive = new \ZipArchive();
        $ZipArchive->open($subDir . $jsonZip);
        $ZipArchive->extractTo($subDir . $dataDir);
        $ZipArchive->close();

        $CompanyReport->setDataUrl('/reports/group_' . $Group->getId() . '/round_' . $CurrentRound->getId() . '/' . (($extendedReport) ? '1/' : '0/'));

        $responseUpload = $this->AuthCommunication->uploadContent($CompanyReport->getProductCode()->getCode(), base64_encode(file_get_contents($subDir . $xmlZip)), md5_file($subDir . $xmlZip), $GroupRoundReportCode->getVersion());
        if(!empty($responseUpload['error']))
        {
            throw new \Exception($responseUpload['error']['message'], $responseUpload['error']['code']);
        }

        $userSsoIds = array();
        foreach($Group->getUsers() as $User)
        {
            $externalId = $User->getExternalId();
            if(!empty($externalId))
            {
                $userSsoIds[] = $externalId;
            }
        }

        $response = $this->AuthCommunication->connectUsersToProduct($CompanyReport->getProductCode()->getCode(), $userSsoIds);
        if(!empty($response['error']))
        {
            throw new \Exception($response['error']['message'], $response['error']['code']);
        }


        if($this->jappsExport)
        {
            $tmpDir = __DIR__ . '/../../../../svn_japps/input/' . $CompanyReport->getProductCode()->getCode();
            mkdir($tmpDir, 0777);
            $ZipArchive = new \ZipArchive();
            $ZipArchive->open($subDir . $mobileZip);
            $ZipArchive->extractTo($tmpDir);
            $ZipArchive->close();

            system('mv ' . $tmpDir . '/data/* ' . $tmpDir . ' && rm -rf ' . $tmpDir . '/data');
            system('cd ' . $tmpDir . '/../.. && svn -q add input/' . $CompanyReport->getProductCode()->getCode());
        }

    }
}